/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.graphics;

import org.eclipse.swt.*;

public final class Font extends Resource {

	public String fontFamily;
	public String fontStyle;
	public String fontWeight;
	public float fontSize;

Font(Device device) {
	super(device);
}
	
public Font (Device device, FontData fd) {
	super(device);
	init(fd);
	init();
}

public Font (Device device, FontData[] fds) {
	super(device);
	if (fds == null) SWT.error(SWT.ERROR_NULL_ARGUMENT);
	if (fds.length == 0) SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	for (int i=0; i<fds.length; i++) {
		if (fds[i] == null) SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	}
	init(fds[0]);
	init();
}

public Font (Device device, String name, int height, int style) {
	super(device);
	if (name == null) SWT.error(SWT.ERROR_NULL_ARGUMENT);
	init(new FontData (name, height, style));
	init();
}

void destroy () {
	fontFamily = fontStyle = fontWeight = null;
	fontSize = 0;
}

public boolean equals (Object object) {
	if (object == this) return true;
	if (!(object instanceof Font)) return false;
	if (isDisposed()) return false;
	Font font = (Font) object;
	return device == font.device &&
	fontFamily.equals(font.fontFamily) &&
	fontStyle.equals(font.fontStyle) &&
	fontWeight.equals(font.fontWeight) &&
	fontSize == font.fontSize;
}

public FontData[] getFontData () {
	if (isDisposed()) SWT.error(SWT.ERROR_GRAPHIC_DISPOSED);
	FontData data = new FontData();
	data.name = fontFamily;
	//TODO - bad screen resolution
	data.height = fontSize * 72 / 96;
	if (fontStyle.equals("italic")) data.style |= SWT.ITALIC;
	if (fontWeight.equals("bold")) data.style |= SWT.BOLD;
	return new FontData[]{data};
}

public int hashCode () {
	if (isDisposed()) return 0;
	return fontFamily.hashCode() ^ fontStyle.hashCode() ^ fontWeight.hashCode();
}

void init (FontData fd) {
	if (fd == null) SWT.error(SWT.ERROR_NULL_ARGUMENT);
	this.fontFamily = fd.name;
	if ((fd.style & SWT.ITALIC) != 0) {
		this.fontStyle = "italic";
	} else {
		this.fontStyle = "normal";
	}
	if ((fd.style & SWT.BOLD) != 0) {
		this.fontWeight = "bold";
	} else {
		this.fontWeight = "normal";
	}
	//TODO - bad screen resolution
	this.fontSize = fd.height * 96 / 72;
}

public boolean isDisposed () {
	return fontFamily == null;
}

public String toString () {
	if (isDisposed()) return "Font {*DISPOSED*}";
	return "Font {" + fontFamily + "," + fontStyle + "," + fontWeight + "," + fontSize + "}";
}

public static Font flex_new (Device device, String fontFamily, String fontStyle, String fontWeight, float height) {
	Font font = new Font(device);
	font.fontFamily = fontFamily;
	font.fontStyle = fontStyle;
	font.fontWeight = fontWeight;
	font.fontSize = height;
	return font;
}

}
